''=============================================================================
''
''    Easy GL2D
''    By Relminator (Richard Eric M. Lope)
''    http://rel.betterwebber.com
''    
''    An easy to use OpenGL 2d lib
''    As easy as fbgfx (easier IMHO if you use my Texture packer)
''    
''    Can automatically load BMP's supported by BLOAD
''    Adds alpha transparency if you want
''    No external dependencies (only uses OpenGL/GLU and FBGFX)
''    Source license is "use or abuse"
''
''=============================================================================



#include once "gl2d.bi"

namespace GL2D

'' Some globals to makelife easier
'' Kinda safe because access is limited to this
'' module and cannot be accessed outside of this
'' module even with a namespace qualifier
 
dim shared as GLuint current_texture = 0
dim shared as GLuint font_list_base = 0
dim shared as GLuint font_textureID = 0

''=============================================================================
''
''    synchronizes the drawing after vblank
''
''    got this from FB's example file
''    added Landeel's stuff for linux
''    
''=============================================================================
sub vsync_on()
	
	dim swapinterval as function(byval interval as integer) as integer
	dim extensions as string
	
	'' setup opengl and retrieve supported extensions
	screencontrol FB.GET_GL_EXTENSIONS, extensions
	
	if (instr(extensions, "WGL_EXT_swap_control") <> 0) then
	    '' extension supported, retrieve proc address
	    swapinterval = ScreenGLProc("wglSwapIntervalEXT")
	else
		swapinterval = ScreenGLProc("glXSwapIntervalSGI")
	end if

    if (swapinterval <> 0) then
        '' ok, we got it. set opengl to wait for vertical sync on buffer swaps
        swapinterval(1)
    end if

end sub


''=============================================================================
''
''    Starts the 2d drawing
''
''=============================================================================
sub start_2d(byval wid as integer, byval hei as integer)
	
    glMatrixMode(GL_PROJECTION)    
    glPushMatrix()
    glLoadIdentity()
    glOrtho(0, wid, hei, 0, -1, 1)
    glMatrixMode(GL_MODELVIEW)
    glPushMatrix()
    glLoadIdentity()
    glTranslatef(0.375, 0.375, 0)	'' magic trick
    
end sub

''=============================================================================
''
''    Ends ortho drawing for if you want to do a 2d-3d combo engine
''
''=============================================================================
sub end_2d()  
	
    glMatrixMode(GL_PROJECTION)
    glPopMatrix()
    glMatrixMode(GL_MODELVIEW)
    glPopMatrix()
    
end sub

''=============================================================================
''
''    private!! sets up the fontsystem
''
''=============================================================================
private sub font_init()

	
    dim as any ptr image = imagecreate(128,128,RGBA(255,0,255,0))
    
    for x as integer = 0 to 15
        for y as integer = 0 to 15
        	draw string image, (x * 8, y * 8), chr( x + y * 16)
        next
    next

	
	font_textureID = load_image_24bit_alpha( image )
	

    imagedestroy( image )


    font_list_base = glGenLists(256)
    dim as single scale = 1/128
    dim as single w = 8 * scale
    dim as single h = 8 * scale

    glPushMatrix()
    glLoadIdentity()
    for font_loop as integer = 0 to 255
	    
	    dim as single x = (font_loop mod 16) * w
	    dim as single y = fix(font_loop / 16) * h
	    
	    glNewList( font_list_base + font_loop, GL_COMPILE )
	    glBegin( GL_QUADS )
	            glTexCoord2f( x, y + h)
	            glVertex2i( 0, 8 )
	           
	            glTexCoord2f( x + w, y + h )
	            glVertex2i( 8, 8 )
	           
	            glTexCoord2f( x + w,y )
	            glVertex2i( 8, 0 )
	           
	            glTexCoord2f( x, y )
	            glVertex2i( 0, 0 )
	    glEnd()
	    
	    glTranslatef( 8,0,0 )
	    glEndList()
	    
    next font_loop

	glPopMatrix()
	
end sub

''=============================================================================
''
''    Sets up OpenGL for 2d mode
''
''=============================================================================
sub screen_init(byval screen_wid as integer, byval screen_hei as integer, byval flags as integer = 0)
	
	if flags then
		screenres screen_wid, screen_hei, 32, 2, FB.GFX_OPENGL or flags
	else
		screenres screen_wid, screen_hei, 32, 2, FB.GFX_OPENGL
	endif
	
	
	'screen information 
	dim w as integer, h as integer 
	'OpenGL params for gluerspective 
	dim FOVy as double            'Field of view angle in Y 
	dim Aspect as double          'Aspect of screen 
	dim znear as double           'z-near clip distance 
	dim zfar as double            'z-far clip distance 

	'using screen info w and h as params 
	glViewport(0, 0, screen_wid, screen_hei)
	
	'Set current Mode to projection(ie: 3d) 
	glMatrixMode(GL_PROJECTION) 
	
	'Load identity matrix to projection matrix 
	glLoadIdentity() 

	'Set gluPerspective params 
	FOVy = 90/2                                     '45 deg fovy 
	Aspect = screen_wid / screen_hei
	znear = 1                                       'Near clip 
	zfar = 500                                      'far clip 
	
	'use glu Perspective to set our 3d frustum dimension up 
	gluPerspective(FOVy, aspect, znear, zfar) 
	
	'Modelview mode 
	'ie. Matrix that does things to anything we draw 
	'as in lines, points, tris, etc. 
	glMatrixMode(GL_MODELVIEW) 
	'load identity(clean) matrix to modelview 
	glLoadIdentity() 
	
	glShadeModel(GL_SMOOTH)                 'set shading to smooth(try GL_FLAT) 
	glClearColor(0.0, 0.0, 0.0, 1.0)        'set Clear color to BLACK 
	glClearDepth(1.0)                       'Set Depth buffer to 1(z-Buffer) 
	glDisable(GL_DEPTH_TEST)                'Disable Depth Testing so that our z-buffer works 
	
	'compare each incoming pixel z value with the z value present in the depth buffer 
	'LEQUAL means than pixel is drawn if the incoming z value is less than 
	'or equal to the stored z value 
	glDepthFunc(GL_LEQUAL) 
	
	'have one or more material parameters track the current color 
	'Material is your 3d model 
	glEnable(GL_COLOR_MATERIAL) 


    'Enable Texturing 
    glEnable(GL_TEXTURE_2D) 
    

   	'Tell openGL that we want the best possible perspective transform 
	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST) 
	
	'Disable Backface culling
	glDisable (GL_CULL_FACE)
	
	glPolygonMode(GL_FRONT, GL_FILL) 
	
	'' enable blending for transparency 
	glEnable(GL_BLEND)    	    
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA)
	
	glDisable( GL_DEPTH_TEST )
	
	glEnable( GL_ALPHA_TEST )
	glAlphaFunc(GL_GREATER, 0)


	glDisable(GL_STENCIL_TEST)
	glDisable(GL_TEXTURE_1D)
	glDisable(GL_LIGHTING)
	glDisable(GL_LOGIC_OP)
	glDisable(GL_DITHER)
	glDisable(GL_FOG)

	glHint(GL_POINT_SMOOTH_HINT, GL_FASTEST)
	glHint(GL_LINE_SMOOTH_HINT , GL_FASTEST)

	glPointSize( 1 )
	glLineWidth( 1 )
	
	'' set up the font system
	font_init()

	start_2d(screen_wid, screen_hei)
	
end sub


''=============================================================================
''
''    Sets blend mode 
''    E_TRANS = normal transparent
''    E_SOLID = solid with no transparency
''    E_BLENDED = 1:1 blending
''    E_GLOW = ADDITIVE blending
''
''=============================================================================
sub set_blend_mode(byval blend_mode as E_BLEND_MODE)
	
	select case blend_mode
		case E_TRANS
			glDisable(GL_BLEND)    	    
			glEnable(GL_ALPHA_TEST)
		case E_SOLID
			glDisable(GL_BLEND)
			glDisable(GL_ALPHA_TEST)    	    
		case E_BLENDED
			glEnable(GL_BLEND)
			glEnable(GL_ALPHA_TEST)    	    
			glBlendFunc(GL_SRC_ALPHA, GL_ONE)
		case E_GLOW
			glEnable(GL_BLEND)
			glEnable(GL_ALPHA_TEST)    	    
			glBlendFunc(GL_ONE, GL_ONE)
		case E_BLACK
			glEnable(GL_BLEND)
			glEnable(GL_ALPHA_TEST)    	
			glBlendFunc(GL_ZERO,GL_ONE_MINUS_SRC_ALPHA)
		case else	
			glDisable(GL_BLEND)    	    
			glEnable(GL_ALPHA_TEST)
	end select
	
end sub

''=============================================================================
''
''    Enables or disables AA depending on the switch
''
''=============================================================================
sub enable_antialias(byval switch as integer = 1)
	
	if switch then
		glEnable( GL_POINT_SMOOTH )
        glEnable( GL_LINE_SMOOTH  )
	else
		glDisable( GL_POINT_SMOOTH )
        glDisable( GL_LINE_SMOOTH  )
	endIf
	
End Sub

''=============================================================================
''
''    Clears the buffer
''
''=============================================================================
sub clear_screen()
	
	glClear(GL_COLOR_BUFFER_BIT OR GL_DEPTH_BUFFER_BIT)

end sub

''=============================================================================
''
''   gl2dColor should use GL2D_RGBA()     
''
''=============================================================================
sub put_pixel( byval x as integer, byval y as integer, byval gl2dcolor as GLuint )
	
	glDisable( GL_TEXTURE_2D )
	glColor4ubv( cast( GLubyte ptr, @gl2dcolor ) )
	glBegin( GL_POINTS )
		glVertex2i( x, y )
	glEnd()
	glEnable( GL_TEXTURE_2D )
	glColor4ub(255,255,255,255)
	
end sub

''=============================================================================
''
''   gl2dColor should use GL2D_RGBA()     
''
''=============================================================================
sub line_2d( byval x1 as integer, byval y1 as integer, byval x2 as integer, byval y2 as integer, byval gl2dcolor as GLuint )

	glDisable( GL_TEXTURE_2D )
	glColor4ubv( cast( GLubyte ptr, @gl2dcolor ) )

    glBegin( GL_LINES )
		glVertex2i( x1, y1 )
		glVertex2i( x2, y2 )
	glEnd()
	glEnable( GL_TEXTURE_2D )

	glColor4ub(255,255,255,255)
	
end sub


''=============================================================================
''
''   gl2dColor should use GL2D_RGBA()     
''
''=============================================================================
sub box( byval x1 as integer, byval y1 as integer, byval x2 as integer, byval y2 as integer, byval gl2dcolor as GLuint )

	glDisable( GL_TEXTURE_2D )
	glColor4ubv( cast( GLubyte ptr, @gl2dcolor ) )
	
    
	glBegin( GL_LINE_STRIP )
		glVertex2i( x1, y1 )
		glVertex2i( x2, y1 )
		glVertex2i( x2, y2 )
		glVertex2i( x1, y2 )
		glVertex2i( x1, y1 )
	glEnd()
	
	
	glEnable( GL_TEXTURE_2D )
	
	glColor4ub(255,255,255,255)
	
end sub


''=============================================================================
''
''   gl2dColor should use GL2D_RGBA()     
''
''=============================================================================
sub box_filled( byval x1 as integer, byval y1 as integer, byval x2 as integer, byval y2 as integer, byval gl2dcolor as GLuint )

	glDisable( GL_TEXTURE_2D )
	glColor4ubv( cast( GLubyte ptr, @gl2dcolor ) )
    
    x2 += 1
    y2 += 1
    
	glBegin(GL_QUADS)
		
		glVertex2i	(x1,y1)
		glVertex2i	(x1,y2)
		glVertex2i	(x2,y2)
		glVertex2i	(x2,y1)
		
	glEnd()
	
	glEnable( GL_TEXTURE_2D )
	
	glColor4ub(255,255,255,255)
    
end sub


''=============================================================================
''
''   gl2dColor should use GL2D_RGBA()     
''
''=============================================================================
sub box_filled_gradient( byval x1 as integer, byval y1 as integer,_
						 byval x2 as integer, byval y2 as integer,_
						 byval gl2dcolor1 as GLuint,_
						 byval gl2dcolor2 as GLuint,_
						 byval gl2dcolor3 as GLuint,_
						 byval gl2dcolor4 as GLuint )

	glDisable( GL_TEXTURE_2D )
	
    x2 += 1
    y2 += 1
    
	glBegin(GL_QUADS)
		glColor4ubv( cast( GLubyte ptr, @gl2dcolor1 ) )
		glVertex2i	(x1,y1)
		glColor4ubv( cast( GLubyte ptr, @gl2dcolor2 ) )
		glVertex2i	(x1,y2)
		glColor4ubv( cast( GLubyte ptr, @gl2dcolor3 ) )
		glVertex2i	(x2,y2)
		glColor4ubv( cast( GLubyte ptr, @gl2dcolor4 ) )
		glVertex2i	(x2,y1)		
	glEnd()
	
	glEnable( GL_TEXTURE_2D )
	
	glColor4ub(255,255,255,255)
    
end sub

''=============================================================================
''
''   gl2dColor should use GL2D_RGBA()     
''
''=============================================================================
sub triangle( byval x1 as integer, byval y1 as integer, byval x2 as integer, byval y2 as integer,_
			  byval x3 as integer, byval y3 as integer, byval gl2dcolor as GLuint )

	glDisable( GL_TEXTURE_2D )
	glColor4ubv( cast( GLubyte ptr, @gl2dcolor ) )

	glBegin( GL_LINE_STRIP )
		glVertex2i( x1, y1 )
		glVertex2i( x2, y2 )
		glVertex2i( x3, y3 )
		glVertex2i( x1, y1 )
	glEnd()
	
	glEnable( GL_TEXTURE_2D )
	
	glColor4ub(255,255,255,255)
	
end sub

''=============================================================================
''
''   gl2dColor should use GL2D_RGBA()     
''
''=============================================================================
sub triangle_filled( byval x1 as integer, byval y1 as integer, byval x2 as integer, byval y2 as integer,_
			         byval x3 as integer, byval y3 as integer, byval gl2dcolor as GLuint )

	glDisable( GL_TEXTURE_2D )
	glColor4ubv( cast( GLubyte ptr, @gl2dcolor ) )

	glBegin( GL_TRIANGLES )
		glVertex2i( x1, y1 )
		glVertex2i( x2, y2 )
		glVertex2i( x3, y3 )
	glEnd()
	
	glEnable( GL_TEXTURE_2D )
	
	glColor4ub(255,255,255,255)
	
end sub

''=============================================================================
''
''   gl2dColor should use GL2D_RGBA()     
''
''=============================================================================
sub triangle_filled_gradient( byval x1 as integer, byval y1 as integer,_ 
							  byval x2 as integer, byval y2 as integer,_
			                  byval x3 as integer, byval y3 as integer,_
			                  byval gl2dcolor1 as GLuint, byval gl2dcolor2 as GLuint, byval gl2dcolor3 as GLuint )

	glDisable( GL_TEXTURE_2D )
	
	glBegin( GL_TRIANGLES )
		glColor4ubv( cast( GLubyte ptr, @gl2dcolor1 ) )
		glVertex2i( x1, y1 )
		glColor4ubv( cast( GLubyte ptr, @gl2dcolor2 ) )
		glVertex2i( x2, y2 )
		glColor4ubv( cast( GLubyte ptr, @gl2dcolor3 ) )
		glVertex2i( x3, y3 )
	glEnd()
	
	glEnable( GL_TEXTURE_2D )
	
	glColor4ub(255,255,255,255)
	
end sub


''=============================================================================
''
''    Draws a circle 
''
''=============================================================================
sub circle_2d(byval x as integer, byval y as integer, byval radius as integer, byval gl2dcolor as GLuint)

	dim as integer stepsize = 45 * 3.141593/radius
	if stepsize < 1 then stepsize = 1
	if stepsize > 5 then stepsize = 5
	
	
	glDisable( GL_TEXTURE_2D )
	
	glColor4ubv( cast( GLubyte ptr, @gl2dcolor ) )
	glBegin(GL_LINE_LOOP)
	for i as integer = 0 to 359 step stepsize
		dim as single angle = i * 3.141593/180
		glVertex2i(x + cos(angle)* radius, y + sin(angle)* radius)
	next i
	glEnd()

	glEnable( GL_TEXTURE_2D )
	
	glColor4ub(255,255,255,255)
	
end sub

''=============================================================================
''
''    Draws a circle 
''
''=============================================================================
sub circle_2d_filled(byval x as integer, byval y as integer, byval radius as integer, byval gl2dcolor as GLuint )

	dim as integer stepsize = 360 * 3.141593/radius
	
	if stepsize < 1 then stepsize = 1
	if stepsize > 5 then stepsize = 5
	
	glDisable( GL_TEXTURE_2D )
	
	glColor4ubv( cast( GLubyte ptr, @gl2dcolor ) )
	glBegin(GL_TRIANGLE_FAN)
	for i as integer = 0 to 359 step stepsize
		dim as single angle = i * 3.141593/180
		glVertex2i(x + cos(angle)* radius, y + sin(angle)* radius)
	next i
	glEnd()

	glEnable( GL_TEXTURE_2D )
	
	glColor4ub(255,255,255,255)
	
end sub


''=============================================================================
''
''    Draws an ellipse
''    Contributed by Michael "h4tt3n" Nissen 
''    syntax: ellipse(center x, center y, semimajor axis, semiminor axis, angle in radians, color)
''
''=============================================================================

sub ellipse(byval x as single, byval y as single, byval a as single, byval b as single, byval angle as single, byval gl2dcolor as GLuint )
       
	'' these constants decide the graphic quality of the ellipse
	const as single  pi = 4*atn(1)        ''        pi
	const as single  twopi  = 2*pi        ''        two pi (radians in a circle)
	const as integer face_length  = 8     ''        approx. face length in pixels
	const as integer max_faces = 256      ''        maximum number of faces in ellipse
	const as integer min_faces = 16       ''        minimum number of faces in ellipse
	
	'' approx. ellipse circumference (hudson's method)
	dim as single h        = (a-b*a-b)/(a+b*a+b)
	dim as single circumference = 0.25*pi*(a+b)*(3*(1+h*0.25)+1/(1-h*0.25))
	
	'' number of faces in ellipse
	dim as integer num_faces = circumference\face_length
	
	'' clamp number of faces
	if num_faces > max_faces then num_faces = max_faces
	if num_faces < min_faces then num_faces = min_faces
	
	'' keep number of faces divisible by 4
	num_faces -= num_faces mod 4
	
	'' precalc cosine theta
	dim as double s         = sin(twopi/num_faces)
	dim as double c         = cos(twopi/num_faces)
	dim as double xx        = 1
	dim as double yy        = 0
	dim as double xt        = 0
	dim as double ax  = cos(angle)
	dim as double ay  = sin(angle)
	
	
	'' draw ellipse
	glDisable( GL_TEXTURE_2D )	
	glColor4ubv( cast( GLubyte ptr, @gl2dcolor ) )

	glBegin(GL_LINE_LOOP)
	
		for i as integer = 1 to num_faces-1
			xt = xx
			xx = c * xx - s * yy
			yy = s * xt + c * yy
			glvertex2f(x+a*xx*ax-b*yy*ay, y+a*xx*ay+b*yy*ax)
		next
		glVertex2f(x+a*ax, y+a*ay)
	
	glEnd()

	glEnable( GL_TEXTURE_2D )
	
	glColor4ub(255,255,255,255)
 
end sub


''=============================================================================
''
''    Draws an ellipse
''    Contributed by Michael "h4tt3n" Nissen(I added a filled version ;*)) 
''    syntax: ellipse_filled(center x, center y, semimajor axis, semiminor axis, angle in radians, color)
''
''=============================================================================

sub ellipse_filled(byval x as single, byval y as single, byval a as single, byval b as single, byval angle as single, byval gl2dcolor as GLuint )
       
	'' these constants decide the graphic quality of the ellipse
	const as single  pi = 4*atn(1)        ''        pi
	const as single  twopi  = 2*pi        ''        two pi (radians in a circle)
	const as integer face_length  = 8     ''        approx. face length in pixels
	const as integer max_faces = 256      ''        maximum number of faces in ellipse
	const as integer min_faces = 16       ''        minimum number of faces in ellipse
	
	'' approx. ellipse circumference (hudson's method)
	dim as single h        = (a-b*a-b)/(a+b*a+b)
	dim as single circumference = 0.25*pi*(a+b)*(3*(1+h*0.25)+1/(1-h*0.25))
	
	'' number of faces in ellipse
	dim as integer num_faces = circumference\face_length
	
	'' clamp number of faces
	if num_faces > max_faces then num_faces = max_faces
	if num_faces < min_faces then num_faces = min_faces
	
	'' keep number of faces divisible by 4
	num_faces -= num_faces mod 4
	
	'' precalc cosine theta
	dim as double s         = sin(twopi/num_faces)
	dim as double c         = cos(twopi/num_faces)
	dim as double xx        = 1
	dim as double yy        = 0
	dim as double xt        = 0
	dim as double ax  = cos(angle)
	dim as double ay  = sin(angle)
	
	
	'' draw ellipse
	glDisable( GL_TEXTURE_2D )	
	glColor4ubv( cast( GLubyte ptr, @gl2dcolor ) )

	glBegin(GL_TRIANGLE_FAN)
	
		for i as integer = 1 to num_faces-1
			xt = xx
			xx = c * xx - s * yy
			yy = s * xt + c * yy
			glvertex2f(x+a*xx*ax-b*yy*ay, y+a*xx*ay+b*yy*ax)
		next
		glVertex2f(x+a*ax, y+a*ay)
	
	glEnd()

	glEnable( GL_TEXTURE_2D )
	
	glColor4ub(255,255,255,255)
 
end sub

''=============================================================================
''
''    Draws a 2d sprite 
''
''=============================================================================
sub sprite( byval x as integer, byval y as integer, byref image as const Tglsprite)
	
	dim  as integer x1 = x
	dim  as integer y1 = y
	dim  as integer x2 = x + (image.width)
	dim  as integer y2 = y + (image.height)
	
	
	'' Only change active texture when there is a need
	'' Speeds up the rendering by batching textures
	if ( image.textureID <> current_texture ) then
		glBindTexture(GL_TEXTURE_2D, image.textureID)
		current_texture = image.textureID
	endif
	
	glBegin(GL_QUADS)
		
		glTexcoord2f(image.u_off, image.v_off)
		glVertex2i	(x1,y1)
		
		glTexcoord2f(image.u_off, image.v_off + image.v_height)
		glVertex2i	(x1,y2)
		
		glTexcoord2f(image.u_off + image.u_width, image.v_off + image.v_height)
		glVertex2i	(x2,y2)
		
		glTexcoord2f(image.u_off + image.u_width, image.v_off)
		glVertex2i	(x2,y1)
		
	glEnd()
	
end sub

''=============================================================================
''
''    Draws a flipped 2d sprite 
''
''=============================================================================
sub sprite_flip( byval x as integer, byval y as integer, byval mode as E_FLIP_MODE, byref image as const Tglsprite)
	
	dim  as integer x1 = x
	dim  as integer y1 = y
	dim  as integer x2 = x + (image.width)
	dim  as integer y2 = y + (image.height)
	
	
	'' Only change active texture when there is a need
	'' Speeds up the rendering by batching textures
	if ( image.textureID <> current_texture ) then
		glBindTexture(GL_TEXTURE_2D, image.textureID)
		current_texture = image.textureID
	endif
	
	dim as single u1, v1, u2, v2 
	select case mode
		case E_NONE:	
			u1 = image.u_off				 : v1 = image.v_off
			u2 = image.u_off + image.u_width : v2 = image.v_off + image.v_height
		case E_V:
			u1 = image.u_off				 : v1 = image.v_off + image.v_height
			u2 = image.u_off + image.u_width : v2 = image.v_off
		case E_H:
			u1 = image.u_off + image.u_width : v1 = image.v_off
			u2 = image.u_off  				 : v2 = image.v_off + image.v_height
		case E_VH:
			u1 = image.u_off + image.u_width : v1 = image.v_off + image.v_height
			u2 = image.u_off				 : v2 = image.v_off 
		case else
			u1 = image.u_off				 : v1 = image.v_off
			u2 = image.u_off + image.u_width : v2 = image.v_off + image.v_height
	end select
	
	glBegin(GL_QUADS)
		
		glTexcoord2f(u1, v1)
		glVertex2i	(x1,y1)
		
		glTexcoord2f(u1, v2)
		glVertex2i	(x1,y2)
		
		glTexcoord2f(u2, v2)
		glVertex2i	(x2,y2)
		
		glTexcoord2f(u2, v1)
		glVertex2i	(x2,y1)
		
	glEnd()
	
end sub

''=============================================================================
''
''    Draws a scaled 2d sprite 
''    having a scale of 1.0 gives you the original size
''
''=============================================================================
sub sprite_scale( byval x as integer, byval y as integer, byval scale as single, byref image as const Tglsprite)
	
	dim  as integer x1 = 0
	dim  as integer y1 = 0
	dim  as integer x2 = (image.width)
	dim  as integer y2 = (image.height)
	
	
	'' Only change active texture when there is a need
	'' Speeds up the rendering by batching textures
	if ( image.textureID <> current_texture ) then
		glBindTexture(GL_TEXTURE_2D, image.textureID)
		current_texture = image.textureID
	endif
	
	glPushMatrix()
	
	glTranslatef(x, y, 0)
	
	glScalef(scale,scale,1.0)
	
	glBegin(GL_QUADS)
		
		glTexcoord2f(image.u_off, image.v_off)
		glVertex2i	(x1,y1)
		
		glTexcoord2f(image.u_off, image.v_off + image.v_height)
		glVertex2i	(x1,y2)
		
		glTexcoord2f(image.u_off + image.u_width, image.v_off + image.v_height)
		glVertex2i	(x2,y2)
		
		glTexcoord2f(image.u_off + image.u_width, image.v_off)
		glVertex2i	(x2,y1)
		
	glEnd()
	
	glPopMatrix()
	
end sub

''=============================================================================
''
''    Draws a center-rotated 2d sprite 
''
''=============================================================================
sub sprite_rotate( byval x as integer, byval y as integer, byval angle as integer,  byref image as const Tglsprite)
	
	dim as integer s_half_x = image.width\2
	dim as integer s_half_y = image.height\2
	
	dim as integer x1 =  -s_half_x
	dim as integer y1 =  -s_half_y
	
	dim as integer x2 =  s_half_x
	dim as integer y2 =  s_half_y
	
	
	'' Only change active texture when there is a need
	'' Speeds up the rendering by batching textures
	if ( image.textureID <> current_texture ) then
		glBindTexture(GL_TEXTURE_2D, image.textureID)
		current_texture = image.textureID
	endif
	
	
	
	
	glPushMatrix()
	
	glTranslatef(x, y, 0)
	glRotatef(angle,0,0,1)
	
	glBegin(GL_QUADS)
		
		glTexcoord2f(image.u_off, image.v_off)
		glVertex2i	(x1,y1)
		
		glTexcoord2f(image.u_off, image.v_off + image.v_height)
		glVertex2i	(x1,y2)
		
		glTexcoord2f(image.u_off + image.u_width, image.v_off + image.v_height)
		glVertex2i	(x2,y2)
		
		glTexcoord2f(image.u_off + image.u_width, image.v_off)
		glVertex2i	(x2,y1)
		
	glEnd()
	
	glPopmatrix()
	
end sub

''=============================================================================
''
''    Draws a center-rotated and scaled 2d sprite 
''
''=============================================================================
sub sprite_rotate_scale( byval x as integer, byval y as integer, byval angle as integer, byval scale as single, byref image as const Tglsprite)
	
	dim as integer s_half_x = image.width\2
	dim as integer s_half_y = image.height\2
	
	dim as integer x1 =  -s_half_x
	dim as integer y1 =  -s_half_y
	
	dim as integer x2 =  s_half_x
	dim as integer y2 =  s_half_y
	
	
	'' Only change active texture when there is a need
	'' Speeds up the rendering by batching textures
	if ( image.textureID <> current_texture ) then
		glBindTexture(GL_TEXTURE_2D, image.textureID)
		current_texture = image.textureID
	endif
	
	
	
	
	glPushMatrix()
	
	glTranslatef(x, y, 0)
	glScalef(scale,scale,1.0)
	glRotatef(angle,0,0,1)
	
	glBegin(GL_QUADS)
		
		glTexcoord2f(image.u_off, image.v_off)
		glVertex2i	(x1,y1)
		
		glTexcoord2f(image.u_off, image.v_off + image.v_height)
		glVertex2i	(x1,y2)
		
		glTexcoord2f(image.u_off + image.u_width, image.v_off + image.v_height)
		glVertex2i	(x2,y2)
		
		glTexcoord2f(image.u_off + image.u_width, image.v_off)
		glVertex2i	(x2,y1)
		
	glEnd()
	
	glPopmatrix()
	
end sub


''=============================================================================
''
''    Draws a stretched 2d sprite 
''
''=============================================================================
sub sprite_stretch( byval x1 as integer, byval y1 as integer,_
					byval x2 as integer, byval y2 as integer,_
					byref image as const Tglsprite)
	
	
	'' Only change active texture when there is a need
	'' Speeds up the rendering by batching textures
	if ( image.textureID <> current_texture ) then
		glBindTexture(GL_TEXTURE_2D, image.textureID)
		current_texture = image.textureID
	endif
	
	glBegin(GL_QUADS)
		
		glTexcoord2f(image.u_off, image.v_off)
		glVertex2i	(x1,y1)
		
		glTexcoord2f(image.u_off, image.v_off + image.v_height)
		glVertex2i	(x1,y2)
		
		glTexcoord2f(image.u_off + image.u_width, image.v_off + image.v_height)
		glVertex2i	(x2,y2)
		
		glTexcoord2f(image.u_off + image.u_width, image.v_off)
		glVertex2i	(x2,y1)
		
	glEnd()
	
end sub


''=============================================================================
''
''    Draws a quad-stretched 2d sprite 
''
''=============================================================================
sub sprite_stretch_on_quad( byval x1 as integer, byval y1 as integer,_
							byval x2 as integer, byval y2 as integer,_
							byval x3 as integer, byval y3 as integer,_
							byval x4 as integer, byval y4 as integer,_
							byref image as const Tglsprite)
	
	
	'' Only change active texture when there is a need
	'' Speeds up the rendering by batching textures
	if ( image.textureID <> current_texture ) then
		glBindTexture(GL_TEXTURE_2D, image.textureID)
		current_texture = image.textureID
	endif
	
	glBegin(GL_QUADS)
		
		glTexcoord2f(image.u_off, image.v_off)
		glVertex2i	(x1,y1)
		
		glTexcoord2f(image.u_off, image.v_off + image.v_height)
		glVertex2i	(x2,y2)
		
		glTexcoord2f(image.u_off + image.u_width, image.v_off + image.v_height)
		glVertex2i	(x3,y3)
		
		glTexcoord2f(image.u_off + image.u_width, image.v_off)
		glVertex2i	(x4,y4)
		
	glEnd()
	
end sub

sub sprite_stretch_h( byval x as integer, byval y as integer, byval length as integer, byref image as const Tglsprite )
	
	
	dim as integer hw = (image.width\2)
	dim as single su = image.u_width/2
	
	dim as integer x1 = x
	dim as integer y1 = y
	dim as integer x2 = x + length
	dim as integer y2 = y + image.height
	
	
	'' Only change active texture when there is a need
	'' Speeds up the rendering by batching textures
	if ( image.textureID <> current_texture ) then
		glBindTexture(GL_TEXTURE_2D, image.textureID)
		current_texture = image.textureID
	endif

	'' left
	dim as integer x2l = x1 + hw
	glBegin(GL_QUADS)
	
		glTexcoord2f(image.u_off, image.v_off)
		glVertex2i(x1,y1)
		
		glTexcoord2f(image.u_off, image.v_off + image.v_height)
		glVertex2i(x1,y2)
		
		glTexcoord2f(image.u_off + su, image.v_off + image.v_height)
		glVertex2i(x2l,y2)
		
		glTexcoord2f(image.u_off + su, image.v_off)
		glVertex2i(x2l,y1)
		
	glEnd()
	
	'' center
	dim as integer x1l = x + hw
	x2l = x2 - hw -1 
	glBegin(GL_QUADS)
	
		glTexcoord2f(image.u_off + su, image.v_off)
		glVertex2i(x1l,y1)
				
		glTexcoord2f(image.u_off + su, image.v_off + image.v_height)
		glVertex2i(x1l,y2)
		
		glTexcoord2f(image.u_off + su, image.v_off + image.v_height)
		glVertex2i(x2l,y2)
		
		glTexcoord2f(image.u_off + su, image.v_off)
		glVertex2i(x2l,y1)
		
	glEnd()
	
	'' right
	x1l = x2 - hw -1
	glBegin(GL_QUADS)
	
		glTexcoord2f(image.u_off + su, image.v_off)
		glVertex2i(x1l,y1)
		
		glTexcoord2f(image.u_off + su, image.v_off + image.v_height)
		glVertex2i(x1l,y2)
		
		glTexcoord2f(image.u_off + image.u_width, image.v_off + image.v_height)
		glVertex2i(x2,y2)
		
		glTexcoord2f(image.u_off + image.u_width, image.v_off)
		glVertex2i(x2,y1)
		
	glEnd()
	
end sub


''=============================================================================
''
''    special function to get a particle texture 
''
''=============================================================================
private function get_glow_image() as GLuint
	
	static as GLuint textureID = 0
	 
	if textureID = 0 then
		
		const IMAGE_WIDTH = 32, IMAGE_HEIGHT = 32, IMAGE_BITDEPTH = 24
		dim as uinteger image_array(0 to 1031) => { _
	    &H00000007, &H00000004, &H00000020, &H00000020, &H00000080, &H00000000, _
	    &H00000000, &H00000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF010101, &HFF010101, &HFF020202, _
	    &HFF020202, &HFF030303, &HFF030303, &HFF030303, &HFF020202, &HFF020202, _
	    &HFF010101, &HFF010101, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF010101, &HFF020202, &HFF030303, _
	    &HFF040404, &HFF050505, &HFF060606, &HFF060606, &HFF060606, &HFF060606, _
	    &HFF060606, &HFF050505, &HFF040404, &HFF030303, &HFF020202, &HFF010101, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF010101, &HFF020202, &HFF040404, _
	    &HFF050505, &HFF070707, &HFF080808, &HFF090909, &HFF0A0A0A, &HFF0B0B0B, _
	    &HFF0B0B0B, &HFF0B0B0B, &HFF0A0A0A, &HFF090909, &HFF080808, &HFF070707, _
	    &HFF050505, &HFF040404, &HFF020202, &HFF010101, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF010101, &HFF030303, _
	    &HFF050505, &HFF070707, &HFF090909, &HFF0B0B0B, &HFF0C0C0C, &HFF0E0E0E, _
	    &HFF0F0F0F, &HFF0F0F0F, &HFF101010, &HFF0F0F0F, &HFF0F0F0F, &HFF0E0E0E, _
	    &HFF0C0C0C, &HFF0B0B0B, &HFF090909, &HFF070707, &HFF050505, &HFF030303, _
	    &HFF010101, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF010101, _
	    &HFF030303, &HFF060606, &HFF080808, &HFF0B0B0B, &HFF0D0D0D, &HFF0F0F0F, _
	    &HFF111111, &HFF131313, &HFF141414, &HFF151515, &HFF151515, &HFF151515, _
	    &HFF141414, &HFF131313, &HFF111111, &HFF0F0F0F, &HFF0D0D0D, &HFF0B0B0B, _
	    &HFF080808, &HFF060606, &HFF030303, &HFF010101, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF010101, &HFF030303, &HFF060606, &HFF090909, &HFF0C0C0C, &HFF0E0E0E, _
	    &HFF121212, &HFF141414, &HFF161616, &HFF181818, &HFF191919, &HFF1A1A1A, _
	    &HFF1B1B1B, &HFF1A1A1A, &HFF191919, &HFF181818, &HFF161616, &HFF141414, _
	    &HFF121212, &HFF0E0E0E, &HFF0C0C0C, &HFF090909, &HFF060606, &HFF030303, _
	    &HFF010101, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF010101, &HFF030303, &HFF060606, &HFF090909, &HFF0C0C0C, _
	    &HFF0F0F0F, &HFF131313, &HFF161616, &HFF191919, &HFF1B1B1B, &HFF1E1E1E, _
	    &HFF1F1F1F, &HFF202020, &HFF212121, &HFF202020, &HFF1F1F1F, &HFF1E1E1E, _
	    &HFF1B1B1B, &HFF191919, &HFF161616, &HFF131313, &HFF0F0F0F, &HFF0C0C0C, _
	    &HFF090909, &HFF060606, &HFF030303, &HFF010101, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF020202, &HFF050505, &HFF080808, _
	    &HFF0C0C0C, &HFF0F0F0F, &HFF131313, &HFF171717, &HFF1B1B1B, &HFF1E1E1E, _
	    &HFF212121, &HFF232323, &HFF252525, &HFF262626, &HFF272727, &HFF262626, _
	    &HFF252525, &HFF232323, &HFF212121, &HFF1E1E1E, &HFF1B1B1B, &HFF171717, _
	    &HFF131313, &HFF0F0F0F, &HFF0C0C0C, &HFF080808, &HFF050505, &HFF020202, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF010101, &HFF040404, _
	    &HFF070707, &HFF0B0B0B, &HFF0E0E0E, &HFF131313, &HFF171717, &HFF1B1B1B, _
	    &HFF1F1F1F, &HFF232323, &HFF262626, &HFF292929, &HFF2B2B2B, &HFF2C2C2C, _
	    &HFF2C2C2C, &HFF2C2C2C, &HFF2B2B2B, &HFF292929, &HFF262626, &HFF232323, _
	    &HFF1F1F1F, &HFF1B1B1B, &HFF171717, &HFF131313, &HFF0E0E0E, &HFF0B0B0B, _
	    &HFF070707, &HFF040404, &HFF010101, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF020202, &HFF050505, &HFF090909, &HFF0D0D0D, &HFF121212, &HFF161616, _
	    &HFF1B1B1B, &HFF1F1F1F, &HFF242424, &HFF282828, &HFF2B2B2B, &HFF2E2E2E, _
	    &HFF313131, &HFF323232, &HFF323232, &HFF323232, &HFF313131, &HFF2E2E2E, _
	    &HFF2B2B2B, &HFF282828, &HFF242424, &HFF1F1F1F, &HFF1B1B1B, &HFF161616, _
	    &HFF121212, &HFF0D0D0D, &HFF090909, &HFF050505, &HFF020202, &HFF000000, _
	    &HFF000000, &HFF010101, &HFF030303, &HFF070707, &HFF0B0B0B, &HFF0F0F0F, _
	    &HFF141414, &HFF191919, &HFF1E1E1E, &HFF232323, &HFF282828, &HFF2C2C2C, _
	    &HFF303030, &HFF333333, &HFF363636, &HFF383838, &HFF383838, &HFF383838, _
	    &HFF363636, &HFF333333, &HFF303030, &HFF2C2C2C, &HFF282828, &HFF232323, _
	    &HFF1E1E1E, &HFF191919, &HFF141414, &HFF0F0F0F, &HFF0B0B0B, &HFF070707, _
	    &HFF030303, &HFF010101, &HFF000000, &HFF010101, &HFF040404, &HFF080808, _
	    &HFF0C0C0C, &HFF111111, &HFF161616, &HFF1B1B1B, &HFF212121, &HFF262626, _
	    &HFF2B2B2B, &HFF303030, &HFF343434, &HFF383838, &HFF3B3B3B, &HFF3D3D3D, _
	    &HFF3D3D3D, &HFF3D3D3D, &HFF3B3B3B, &HFF383838, &HFF343434, &HFF303030, _
	    &HFF2B2B2B, &HFF262626, &HFF212121, &HFF1B1B1B, &HFF161616, &HFF111111, _
	    &HFF0C0C0C, &HFF080808, &HFF040404, &HFF010101, &HFF000000, &HFF020202, _
	    &HFF050505, &HFF090909, &HFF0E0E0E, &HFF131313, &HFF181818, &HFF1E1E1E, _
	    &HFF232323, &HFF292929, &HFF2E2E2E, &HFF333333, &HFF383838, &HFF3C3C3C, _
	    &HFF3F3F3F, &HFF424242, &HFF424242, &HFF424242, &HFF3F3F3F, &HFF3C3C3C, _
	    &HFF383838, &HFF333333, &HFF2E2E2E, &HFF292929, &HFF232323, &HFF1E1E1E, _
	    &HFF181818, &HFF131313, &HFF0E0E0E, &HFF090909, &HFF050505, &HFF020202, _
	    &HFF000000, &HFF020202, &HFF060606, &HFF0A0A0A, &HFF0F0F0F, &HFF141414, _
	    &HFF191919, &HFF1F1F1F, &HFF252525, &HFF2B2B2B, &HFF313131, &HFF363636, _
	    &HFF3B3B3B, &HFF3F3F3F, &HFF434343, &HFF454545, &HFF474747, &HFF454545, _
	    &HFF434343, &HFF3F3F3F, &HFF3B3B3B, &HFF363636, &HFF313131, &HFF2B2B2B, _
	    &HFF252525, &HFF1F1F1F, &HFF191919, &HFF141414, &HFF0F0F0F, &HFF0A0A0A, _
	    &HFF060606, &HFF020202, &HFF000000, &HFF030303, &HFF060606, &HFF0B0B0B, _
	    &HFF0F0F0F, &HFF151515, &HFF1A1A1A, &HFF202020, &HFF262626, &HFF2C2C2C, _
	    &HFF323232, &HFF383838, &HFF3D3D3D, &HFF424242, &HFF454545, &HFFAEAEAE, _
	    &HFFD3D3D3, &HFF454545, &HFF454545, &HFF424242, &HFF3D3D3D, &HFF383838, _
	    &HFF323232, &HFF2C2C2C, &HFF262626, &HFF202020, &HFF1A1A1A, &HFF151515, _
	    &HFF0F0F0F, &HFF0B0B0B, &HFF060606, &HFF030303, &HFF000000, &HFF030303, _
	    &HFF060606, &HFF0B0B0B, &HFF101010, &HFF151515, &HFF1B1B1B, &HFF212121, _
	    &HFF272727, &HFF2C2C2C, &HFF323232, &HFF383838, &HFF3D3D3D, &HFF424242, _
	    &HFF474747, &HFFD3D3D3, &HFFFFFFFF, &HFF454545, &HFF474747, &HFF424242, _
	    &HFF3D3D3D, &HFF383838, &HFF323232, &HFF2C2C2C, &HFF272727, &HFF212121, _
	    &HFF1B1B1B, &HFF151515, &HFF101010, &HFF0B0B0B, &HFF060606, &HFF030303, _
	    &HFF000000, &HFF030303, &HFF060606, &HFF0B0B0B, &HFF0F0F0F, &HFF151515, _
	    &HFF1A1A1A, &HFF202020, &HFF262626, &HFF2C2C2C, &HFF323232, &HFF383838, _
	    &HFF3D3D3D, &HFF424242, &HFF454545, &HFF454545, &HFF454545, &HFF454545, _
	    &HFF454545, &HFF424242, &HFF3D3D3D, &HFF383838, &HFF323232, &HFF2C2C2C, _
	    &HFF262626, &HFF202020, &HFF1A1A1A, &HFF151515, &HFF0F0F0F, &HFF0B0B0B, _
	    &HFF060606, &HFF030303, &HFF000000, &HFF020202, &HFF060606, &HFF0A0A0A, _
	    &HFF0F0F0F, &HFF141414, &HFF191919, &HFF1F1F1F, &HFF252525, &HFF2B2B2B, _
	    &HFF313131, &HFF363636, &HFF3B3B3B, &HFF3F3F3F, &HFF434343, &HFF454545, _
	    &HFF474747, &HFF454545, &HFF434343, &HFF3F3F3F, &HFF3B3B3B, &HFF363636, _
	    &HFF313131, &HFF2B2B2B, &HFF252525, &HFF1F1F1F, &HFF191919, &HFF141414, _
	    &HFF0F0F0F, &HFF0A0A0A, &HFF060606, &HFF020202, &HFF000000, &HFF020202, _
	    &HFF050505, &HFF090909, &HFF0E0E0E, &HFF131313, &HFF181818, &HFF1E1E1E, _
	    &HFF232323, &HFF292929, &HFF2E2E2E, &HFF333333, &HFF383838, &HFF3C3C3C, _
	    &HFF3F3F3F, &HFF424242, &HFF424242, &HFF424242, &HFF3F3F3F, &HFF3C3C3C, _
	    &HFF383838, &HFF333333, &HFF2E2E2E, &HFF292929, &HFF232323, &HFF1E1E1E, _
	    &HFF181818, &HFF131313, &HFF0E0E0E, &HFF090909, &HFF050505, &HFF020202, _
	    &HFF000000, &HFF010101, &HFF040404, &HFF080808, &HFF0C0C0C, &HFF111111, _
	    &HFF161616, &HFF1B1B1B, &HFF212121, &HFF262626, &HFF2B2B2B, &HFF303030, _
	    &HFF343434, &HFF383838, &HFF3B3B3B, &HFF3D3D3D, &HFF3D3D3D, &HFF3D3D3D, _
	    &HFF3B3B3B, &HFF383838, &HFF343434, &HFF303030, &HFF2B2B2B, &HFF262626, _
	    &HFF212121, &HFF1B1B1B, &HFF161616, &HFF111111, &HFF0C0C0C, &HFF080808, _
	    &HFF040404, &HFF010101, &HFF000000, &HFF010101, &HFF030303, &HFF070707, _
	    &HFF0B0B0B, &HFF0F0F0F, &HFF141414, &HFF191919, &HFF1E1E1E, &HFF232323, _
	    &HFF282828, &HFF2C2C2C, &HFF303030, &HFF333333, &HFF363636, &HFF383838, _
	    &HFF383838, &HFF383838, &HFF363636, &HFF333333, &HFF303030, &HFF2C2C2C, _
	    &HFF282828, &HFF232323, &HFF1E1E1E, &HFF191919, &HFF141414, &HFF0F0F0F, _
	    &HFF0B0B0B, &HFF070707, &HFF030303, &HFF010101, &HFF000000, &HFF000000, _
	    &HFF020202, &HFF050505, &HFF090909, &HFF0D0D0D, &HFF121212, &HFF161616, _
	    &HFF1B1B1B, &HFF1F1F1F, &HFF242424, &HFF282828, &HFF2B2B2B, &HFF2E2E2E, _
	    &HFF313131, &HFF323232, &HFF323232, &HFF323232, &HFF313131, &HFF2E2E2E, _
	    &HFF2B2B2B, &HFF282828, &HFF242424, &HFF1F1F1F, &HFF1B1B1B, &HFF161616, _
	    &HFF121212, &HFF0D0D0D, &HFF090909, &HFF050505, &HFF020202, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF010101, &HFF040404, &HFF070707, &HFF0B0B0B, _
	    &HFF0E0E0E, &HFF131313, &HFF171717, &HFF1B1B1B, &HFF1F1F1F, &HFF232323, _
	    &HFF262626, &HFF292929, &HFF2B2B2B, &HFF2C2C2C, &HFF2C2C2C, &HFF2C2C2C, _
	    &HFF2B2B2B, &HFF292929, &HFF262626, &HFF232323, &HFF1F1F1F, &HFF1B1B1B, _
	    &HFF171717, &HFF131313, &HFF0E0E0E, &HFF0B0B0B, &HFF070707, &HFF040404, _
	    &HFF010101, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF020202, _
	    &HFF050505, &HFF080808, &HFF0C0C0C, &HFF0F0F0F, &HFF131313, &HFF171717, _
	    &HFF1B1B1B, &HFF1E1E1E, &HFF212121, &HFF232323, &HFF252525, &HFF262626, _
	    &HFF272727, &HFF262626, &HFF252525, &HFF232323, &HFF212121, &HFF1E1E1E, _
	    &HFF1B1B1B, &HFF171717, &HFF131313, &HFF0F0F0F, &HFF0C0C0C, &HFF080808, _
	    &HFF050505, &HFF020202, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF010101, &HFF030303, &HFF060606, &HFF090909, &HFF0C0C0C, _
	    &HFF0F0F0F, &HFF131313, &HFF161616, &HFF191919, &HFF1B1B1B, &HFF1E1E1E, _
	    &HFF1F1F1F, &HFF202020, &HFF212121, &HFF202020, &HFF1F1F1F, &HFF1E1E1E, _
	    &HFF1B1B1B, &HFF191919, &HFF161616, &HFF131313, &HFF0F0F0F, &HFF0C0C0C, _
	    &HFF090909, &HFF060606, &HFF030303, &HFF010101, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF010101, &HFF030303, _
	    &HFF060606, &HFF090909, &HFF0C0C0C, &HFF0E0E0E, &HFF121212, &HFF141414, _
	    &HFF161616, &HFF181818, &HFF191919, &HFF1A1A1A, &HFF1B1B1B, &HFF1A1A1A, _
	    &HFF191919, &HFF181818, &HFF161616, &HFF141414, &HFF121212, &HFF0E0E0E, _
	    &HFF0C0C0C, &HFF090909, &HFF060606, &HFF030303, &HFF010101, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF010101, &HFF030303, &HFF060606, &HFF080808, &HFF0B0B0B, _
	    &HFF0D0D0D, &HFF0F0F0F, &HFF111111, &HFF131313, &HFF141414, &HFF151515, _
	    &HFF151515, &HFF151515, &HFF141414, &HFF131313, &HFF111111, &HFF0F0F0F, _
	    &HFF0D0D0D, &HFF0B0B0B, &HFF080808, &HFF060606, &HFF030303, &HFF010101, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF010101, &HFF030303, _
	    &HFF050505, &HFF070707, &HFF090909, &HFF0B0B0B, &HFF0C0C0C, &HFF0E0E0E, _
	    &HFF0F0F0F, &HFF0F0F0F, &HFF101010, &HFF0F0F0F, &HFF0F0F0F, &HFF0E0E0E, _
	    &HFF0C0C0C, &HFF0B0B0B, &HFF090909, &HFF070707, &HFF050505, &HFF030303, _
	    &HFF010101, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF010101, &HFF020202, &HFF040404, &HFF050505, &HFF070707, _
	    &HFF080808, &HFF090909, &HFF0A0A0A, &HFF0B0B0B, &HFF0B0B0B, &HFF0B0B0B, _
	    &HFF0A0A0A, &HFF090909, &HFF080808, &HFF070707, &HFF050505, &HFF040404, _
	    &HFF020202, &HFF010101, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF010101, _
	    &HFF020202, &HFF030303, &HFF040404, &HFF050505, &HFF060606, &HFF060606, _
	    &HFF060606, &HFF060606, &HFF060606, &HFF050505, &HFF040404, &HFF030303, _
	    &HFF020202, &HFF010101, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF010101, &HFF010101, &HFF020202, _
	    &HFF020202, &HFF030303, &HFF030303, &HFF030303, &HFF020202, &HFF020202, _
	    &HFF010101, &HFF010101, &HFF000000, &HFF000000, &HFF000000, &HFF000000, _
	    &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000, &HFF000000 }

	
		textureID = load_image( @image_array(0),GL_LINEAR )
	
	endif
	
	return textureID 

end function

''=============================================================================
''
''    Draws a 2d glowing line
''    x1,y1,x2,y2 as start and end coordinate of the line
''    lwidth is the width of the line (how fat it is)
''    mycolor should use GL2D_RGBA() 
''
''=============================================================================

sub line_glow ( byval x1 as single, byval y1 as single, byval x2 as single, byval y2 as single,_
				byval lwidth as single, byval mycolor as GLuint)

	
	
	dim as GLuint textureID = get_glow_image()
	
	'' Only change active texture when there is a need
	'' Speeds up the rendering by batching textures
	if ( textureID <> current_texture ) then
		glBindTexture(GL_TEXTURE_2D, textureID)
		current_texture = textureID
	endif

	
	dim as single nx,ny	
	nx = -(y2-y1)
	ny =  (x2-x1)
	
	dim leng as single
    leng = sqr(nx * nx + ny * ny )
    nx = nx / leng
    ny = ny / leng

	nx *= lwidth/2
	ny *= lwidth/2
	 	
	dim as single lx1, ly1, lx2, ly2, lx3, ly3, lx4, ly4
	
	lx1 = x2+nx
    ly1 = y2+ny
    lx2 = x2-nx
    ly2 = y2-ny                            
    lx3 = x1-nx
    ly3 = y1-ny
    lx4 = x1+nx
    ly4 = y1+ny 

	glColor4ubv (cast(GLubyte ptr, @mycolor))

	''MAIN
	glbegin(GL_QUADS)
		glTexCoord2f( 0.5,0 )
		glVertex3f( lx1,ly1,0 )

		glTexCoord2f( 0.5,1 )
		glVertex3f( lx2,ly2,0 )

		glTexCoord2f( 0.5,1 )
		glVertex3f( lx3, ly3,0 )

		glTexCoord2f( 0.5,0 )
		glVertex3f( lx4,ly4,0 )
	glend()

	'RIGHT
	dim as single lx5, ly5,lx6,ly6,vx,vy
	vx = (x2-x1)
	vy = (y2-y1)
	leng = sqr(vx * vx + vy * vy )
	vx = vx / leng
	vy = vy / leng
	vx *= lwidth/2
	vy *= lwidth/2
	
	lx5 = lx1 + vx
	ly5 = ly1 + vy
	lx6 = lx2 + vx
	ly6 = ly2 + vy
	
	glbegin(GL_QUADS)
		glTexCoord2f( 0.5,0 )
		glVertex3f( lx1,ly1,0 )

		glTexCoord2f( 1,0 )
		glVertex3f( lx5,ly5,0 )

		glTexCoord2f( 1,1 )
		glVertex3f( lx6, ly6,0 )

		glTexCoord2f( 0.5,1 )
		glVertex3f( lx2,ly2,0 )
	glend() 

	'LEFT
	lx5 = lx4 -vx
	ly5 = ly4 -vy
	lx6 = lx3 -vx
	ly6 = ly3 -vy
	glbegin(GL_QUADS)
		glTexCoord2f( 0.5,0 )
		glVertex3f( lx4,ly4,0 )
		
		glTexCoord2f( 0.5,1 )
		glVertex3f( lx3,ly3,0 )

		glTexCoord2f( 1,1 )
		glVertex3f( lx6, ly6,0 )

		glTexCoord2f( 1,0 )
		glVertex3f( lx5,ly5,0 )
		
	glend()
	
end sub


''=============================================================================
''
''    Rudimentary font system
''
''=============================================================================
sub print_scale(byval x as integer, byval y as integer, byval scale as single, byref text as const string)

   
   '' Only change active texture when there is a need
	'' Speeds up the rendering by batching textures
	if ( font_textureID <> current_texture ) then
		glBindTexture(GL_TEXTURE_2D, font_textureID)
		current_texture = font_textureID
	endif
	    
    glPushMatrix()
    glLoadIdentity()
    glTranslatef( x, y, 0 )
    glScalef(scale, scale, 1)
    glListBase( font_list_base )
        for i as integer = 0 to len(text) - 1
        	glCallList( font_list_base + text[i] )
        next i
    glPopMatrix()

end sub


''=============================================================================
''
''    Loads a 32 bit buffer (BLOADed from a BMP image)
''
''=============================================================================
function load_image( byval image as any ptr , byval filter_mode as GLuint = GL_NEAREST) as GLuint
	
        dim as GLuint TextureID
   		dim as ubyte r, g, b, a
   		dim As FB.IMAGE ptr temp = image
   		       
        glEnable( GL_TEXTURE_2D )
        glGenTextures(1, @TextureID)
        glBindTexture(GL_TEXTURE_2D, TextureID)
        
   		glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT )
        glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT )
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, filter_mode )
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, filter_mode )
        glTexEnvf( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE )
               
        glTexImage2d( GL_TEXTURE_2D, 0, GL_RGBA, temp->width, temp->height,_
        			  0, GL_BGRA, GL_UNSIGNED_BYTE, image + sizeof(FB.IMAGE) )
        glBindTexture(GL_TEXTURE_2D, 0)
        
        
        return TextureID
        
End function

''=============================================================================
''
''    Loads a 32 bit buffer (BLOADed from a 24-bit BMP image)
''    Adds alpha transparency
''
''=============================================================================
function load_image_24bit_alpha( byval image as any ptr, byval filter_mode as GLuint = GL_NEAREST ) as GLuint
	
        dim as GLuint TextureID
   		dim as ubyte r, g, b, a
   		dim As FB.IMAGE ptr temp = image
   		
   		for y as integer = 0 to temp->height-1
	    	dim as uinteger ptr p = cast(uinteger ptr,(image + sizeof (FB.IMAGE)) + y * temp->pitch)	   
		    for x as integer = 0 to temp->width-1
		    	a = argb_a(p[x])
		    	r = argb_r(p[x])
		    	g = argb_g(p[x])
		    	b = argb_b(p[x])
		    	'' check for transparency
				if ( g = 0 ) then
					if ( (r = 255) and (b = 255) ) then
						p[x] = rgba(r,g,b,0)
					endif
				endif
		    next x
   		next y
   		
   		
   		glEnable( GL_TEXTURE_2D )
        glGenTextures(1, @TextureID)
        glBindTexture(GL_TEXTURE_2D, TextureID)
        
   		glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT )
        glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT )
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, filter_mode )
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, filter_mode )
        glTexEnvf( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE )
               
        glTexImage2d( GL_TEXTURE_2D, 0, GL_RGBA, temp->width, temp->height,_
        			  0, GL_BGRA, GL_UNSIGNED_BYTE, image + sizeof(FB.IMAGE) )
        glBindTexture(GL_TEXTURE_2D, 0)
        
        
        return TextureID
        
end function


''=============================================================================
''
''    Loads a 32 bit buffer (BLOADed from an 8-bit BMP image)
''    Adds alpha transparency
''
''=============================================================================
function load_image_8bit_alpha( byval image as any ptr, byval filter_mode as GLuint = GL_NEAREST ) as GLuint
	
        dim as GLuint TextureID
   		dim as ubyte r, g, b, a
   		dim As FB.IMAGE ptr temp = image
   
		    	
   		for y as integer = 0 to temp->height-1
	    	dim as uinteger ptr p = cast(uinteger ptr,(image + sizeof(FB.IMAGE)) + y * temp->pitch)	   
		    for x as integer = 0 to temp->width-1
		    	a = argb_a(p[x])
		    	r = argb_r(p[x])
		    	g = argb_g(p[x])
		    	b = argb_b(p[x])
		    	'' check for transparency
		    	if ( ( p[x] and &H00FFFFFF ) = 0 ) then
		    		p[x] = rgba(r,g,b,0)
		    	endif
				
		    next x
   		next y
   		
   		
   		glEnable( GL_TEXTURE_2D )
        glGenTextures(1, @TextureID)
        glBindTexture(GL_TEXTURE_2D, TextureID)
        
   		glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT )
        glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT )
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, filter_mode )
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, filter_mode )
        glTexEnvf( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE )
               
        glTexImage2d( GL_TEXTURE_2D, 0, GL_RGBA, temp->width, temp->height,_
        			  0, GL_BGRA, GL_UNSIGNED_BYTE, image + sizeof(FB.IMAGE) )
        glBindTexture(GL_TEXTURE_2D, 0)
        
        
        return TextureID
        
end function

''=============================================================================
''
''    Loads any type of BMP (Power of 2) into an OpenGL texture
''
''=============================================================================
function load_BMP_to_texture(byref filename as string) as GLuint
	
        if filename = "" then return 0
         
        dim as GLuint textureID
        
        dim as integer F = freefile
        dim as integer w, h		'' width and height
        dim as ushort bpp		'' bits per pixel
        
        '' test header
        open filename for binary as #f
        get #f, 19, w
        get #f, 23, h
        get #f, 29, bpp
        close #f
       
        dim as FB.IMAGE ptr image = imagecreate(w,h)
        bload filename, image
       
       
        select case bpp
        	case 8
        		textureID = load_image_8bit_alpha(image)
        	case 24
        		textureID = load_image_24bit_alpha(image)
        	case else
        		textureID = load_image(image)
        end select
        
        imagedestroy image
        
        return textureID
        
end function 



''=============================================================================
''
''    Loads any type of BMP (Power of 2) into an OpenGL texture
''    results are stored in a glsprite datatype
''
''=============================================================================
function load_BMP_to_glsprite(byref filename as string) as Tglsprite
	
        if filename = "" then exit function
        
        dim as Tglsprite glsprite 
        dim as GLuint textureID
        
        dim as integer F = freefile
        dim as integer w, h		'' width and height
        dim as ushort bpp		'' bits per pixel
        
        '' test header
        open filename for binary as #f
        get #f, 19, w
        get #f, 23, h
        get #f, 29, bpp
        close #f
       
        dim as FB.IMAGE ptr image = imagecreate(w,h)
        bload filename, image
       
       
        select case bpp
        	case 8
        		textureID = load_image_8bit_alpha(image)
        	case 24
        		textureID = load_image_24bit_alpha(image)
        	case else
        		textureID = load_image(image)
        end select
        
        
        glsprite.width = w
        glsprite.height = h
        glsprite.u_off = 0.0
        glsprite.v_off = 0.0
        glsprite.u_width = 1.0
        glsprite.v_height = 1.0
        glsprite.textureID = textureID
        
        imagedestroy image
        
        return glsprite
        
end function 


''=============================================================================
''
''    Initializes our spriteset
''    Uses a BI file generated by rel's texture packer
''
''=============================================================================
sub init_sprites(spriteset() as Tglsprite, texcoords() as uinteger,_
				 byval wid as integer, byval hei as integer,_
				 byval textureID as GLuint)
	
	dim as integer max_sprites = ubound(texcoords)\4
	redim spriteset(max_sprites)
	
	'' init sprites texture coords and texture ID
	for i as integer = 0 to max_sprites
		dim as integer  j = i * 4
		dim as integer w = texcoords(j+2)
		dim as integer h = texcoords(j+3)
		spriteset(i).textureID 	= textureID
		spriteset(i).width 		= w
		spriteset(i).height 	= h
		spriteset(i).u_off 		= texcoords(j)/wid
		spriteset(i).v_off 		= texcoords(j+1)/hei 
		spriteset(i).u_width 	= w/wid
		spriteset(i).v_height 	= h/hei 	
	next i
	
end sub


''=============================================================================
''
''    deallocates the texture of a spriteset 
''
''=============================================================================
sub destroy_sprites(spriteset() as Tglsprite)
	
	glDeleteTextures( 1, @spriteset(0).textureID )
	
	for i as integer = 0 to ubound(spriteset)
		spriteset(i).textureID 	= 0
	next i
	
end sub

''=============================================================================
''
''    call this before closing the program
''
''=============================================================================
sub destroy()
	
    glDeleteLists( font_list_base, 256 )
    glDeleteTextures( 1, @font_textureID )
	
end sub


''=============================================================================
''
''    use this to limit the FPS if vsync_ON does not work
''
''=============================================================================
function fps_limit(byval max_FPS as single) as single
	
	'' vars to be used in calculating FPS
	static as integer frames_per_second = 0.0
	static as single last_time = 0.0 
	
	static as single time_start = 0.0
	
	'' time interval between frames
	dim as single time_difference

	static as single fps = 60
		
	
	'' Increase the frame counter
	frames_per_second += 1

	do : loop until (timer - time_start) >= (1/max_FPS)
	
	time_start =  timer
	
	if  (time_start - last_time) > 1.0 then
	 
		'' save old_time
	    last_time = time_start

		'' duh?		
		FPS = frames_per_second
		
		'' Reset the frames counter since 1 second has already elapsed
	   frames_per_second = 0
	   
	 end if
	 	   
	 
	return fps

end function

end namespace



